/************************************************************************************************************\

Module Name:    LAncIn.h

Description:    The LAncIn module of the API provides the programming interface for the ancillary data input
                hardware and its associated input devices.

References:     LAncIn.doc Revision

    Copyright (c) 2022, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LANCIN_H
#define INC_LANCIN_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAnc.h"
#include "LVideoIn.h"

typedef struct LANCIN_OBJECT*      LAncIn_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                                U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Capture ancillary stream on an Ancillary input.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LAncIn.h"

    int main()
    {
        LDevice_Handle              hDevice         = MNULL;
        LBuffer_Handle              hCapturedBuffer = MNULL;
        LAncIn_Handle               hAncIn          = MNULL;
        MUINT32                     uiBufferSizeInBytes = 4096;
        MUINT32                     uiIndex         = 0;;

        Liberatus_Load();

        hDevice = Liberatus_GetDevice(0);           // Primary device handle

        // Get handle on first usable anc input.
        LAncIn_GetHandle(hDevice,
            0,                                      // First anc input using default signal type
            LAccessMode_READWRITE,
            &hAncIn);

        LAncIn_CreateBuffers(hAncIn,
            uiBufferSizeInBytes,
            3);

        // Start capture ancillary.
        LAncIn_StartCapture(hAncIn);

        for(uiIndex = 0;  uiIndex < 100; uiIndex++)
        {
            LAncIn_GetNextBuffer(
                hAncIn,
                MTRUE,                // Wait for Ready buffer.
                200,                  // in ms
                &hCapturedBuffer,
                MNULL,
                MNULL);

            // USE ANCILLARY BUFFER HERE

            LAncIn_ReleaseBuffer(
                hAncIn,
                hCapturedBuffer);

        }

        // End ancillary capture
        LAncIn_StopCapture(hAncIn);

        LAncIn_DestroyBuffers(hAncIn);

        // Release ancillary input.
        LAncIn_ReleaseHandle(hAncIn);

        Liberatus_UnLoad();
    }

*/

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------


/************************************************************************************************************\

    The main purpose of the Ancillary Input module is to provide the functionality to input (capture) an ancillary
    input stream into the Matrox SV2 system.

    The module has services to:
        - enumerate ancillary input paths and their ancillary capabilities
        - configure the available ancillary input path in a specific mode to capture ancillary buffers
        - retrieve the captured ancillary buffers
        - retrieve the state of the interfaces
        - be notified of capture events
        - adjust dynamic controls (such as input gain level)

    Based on the hardware configuration, the module can handle analog and/or digital ancillary input stream
    capture in mono, stereo, or multi-channels. It can also handle PCM (uncompressed) and/or compressed ancillary
    input streams.

    The module can handle multiple ancillary input streams at a time.

\************************************************************************************************************/

/************************************************************************************************************\
    A ancillary inputs can be used to capture ancillary input stream data into LBuffer objects, named capture buffers
    from here on, that can then be used to perform processing on the captured data.

    A ancillary input has an internal set of capture buffers that are used to capture input stream data and
    that are loaned to a Liberatus application for processing.

    Each ancillary input internal capture buffer has 2 different attributes that describe its current state.

    The first attribute is called 'Ready' and it represents whether it contains all captured stream data
    or not. The second attribute is called 'Loaned' and it represents whether a capture buffer is loaned
    to a Liberatus application for processing or not.

    Bellow you will find when a capture buffer 'Ready' and 'Loaned' state attributes change.

    Ready state attribute:
    ======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |     Stream Captured Data            |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |        EMPTY         |                                     |        READY         |
        |                      |<------------------------------------|                      |
        |                      |        ReleaseBuffer() or           |                      |
        +----------------------+      DiscardReadyBuffers() or       +----------------------+
                                     ScheduleReleaseBuffer()

    Loaned state attribute:
    =======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |           GetNextBuffer()           |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |      RELEASED        |                                     |        LOANED        |
        |                      |<------------------------------------|                      |
        |                      |           ReleaseBuffer() or        |                      |
        +----------------------+        ScheduleReleaseBuffer()      +----------------------+

    Attributes states meaning:
    ==========================

    +-------------+-------------+-----------------------------------------------------------+
    |   Ready     |   Loaned    | Meaning                                                   |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MFALSE    | Capture buffer contains no stream data and isn't loaned   |
    |             |             | to a Liberatus application.The ancillary input can capture|
    |             |             | input stream date in the buffer.                          |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MTRUE     | Capture buffer contains no stream data and is loaned to a |
    |             |             | Liberatus application. The ancillary input can capture    |
    |             |             | input stream date in the buffer and the Liberatus         |
    |             |             | application can schedule operations on the capture buffer.|
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MFALSE    | Capture buffer contains stream data and isn't loaned to   |
    |             |             | a Liberatus application.The ancillary input can't capture |
    |             |             | input stream date in the buffer.                          |
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MTRUE     | Capture buffer contains stream data and is loaned to a    |
    |             |             | Liberatus application.  The ancillary input can't capture |
    |             |             | input stream date in the buffer and the Liberatus         |
    |             |             | application can schedule or perform direct operations on  |
    |             |             | the capture buffer.                                       |
    +-------------+-------------+-----------------------------------------------------------+

        Capture Buffer pool definitions

            <Total>        Count of all buffers in the pool.

            <Ready>        Count of buffers containing captured data.
            <Empty>        Count of buffers containing no captured data. <Total - Ready>

            <Loaned>       Count of buffers loaned to the application
            <Released>     Count of buffers not loan to the application. <Total - Loaned>

\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enum:           LAncIn_ControlId

Description:    Enumeration defining all possible ancillary input control type values.

Comments:       - Each ancillary input control has a minimal, a maximal, a default value and also a step value
                  that can be used compute valid value to modify the control.  See the LAncIn_ControlInfo
                  structure or the LAncIn_SetControlValue() service for more details.

\************************************************************************************************************/
typedef enum
{
    LAncIn_ControlId_INVALID                = 0x00000000,   // Invalid value.
    LAncIn_ControlId_FORCE32                = 0x7FFFFFFF,   // Dummy value to force enumeration to use
                                                              // 32-bits.
} LAncIn_ControlId;

/************************************************************************************************************\

Enum:           LAncIn_Event

Description:    Enumerates all possible capture events.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LAncIn_Event_INVALID              = 0x00000000,     // Invalid value.
    LAncIn_Event_START_PENDING        = 0x00000001,     // Capture started but it's waiting on an external
                                                        //  event to start capturing.
    LAncIn_Event_STARTED              = 0x00000002,     // Capture already started, ancillary data
                                                        //  are being captured.
    LAncIn_Event_STOPPED              = 0x00000004,     // Capture has stopped, but it's still
                                                        //  initialized and can be restarted.
    LAncIn_Event_STATE_CHANGED        = 0x00000008,     // Capture state has changed.
    LAncIn_Event_SOURCE_CHANGED       = 0x00000010,     // Input source has changed.  The capture was stopped
                                                        //  and must be re-initialized to capture
                                                        //  ancillary data again.
    LAncIn_Event_OUT_OF_BUFFER        = 0x00000020,     // Capture is forced to drop ancillary data because no
                                                        // empty capture buffer is available. Capture
                                                        //  automatically resumes when capture buffers become
                                                        //  available again.
    LAncIn_Event_ALL_BUFFER_RELEASED  = 0x00000040,     // All allocated capture buffers are currently in the
                                                        //  Released state.  This event can be used to figure
                                                        //  out when the handle to the ancillary input device can
                                                        //  be safely released without causing a resource
                                                        //  leaks.
    LAncIn_Event_SOURCE_DETECTED      = 0x00000080,     // Input source has been detected.

    LAncIn_Event_FORCE32              = 0x7FFFFFFF    // Dummy value to force to use 32-bits.
} LAncIn_Event;

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------


#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LAncIn_GetCount

Description:    Gets number of independent ancillary inputs available for the specified device.

Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to an MUINT32 variable to be filled with the number of independent
                                    ancillary inputs available.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetCount(
                LDevice_Handle      hDev,
                MUINT32*            puiCount);


/************************************************************************************************************\

Function:       LAncIn_GetHandle

Description:    Gets ancillary input handle for the specified independent ancillary input device.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent ancillary input for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phAncIn         Pointer to an LAncIn_Handle variable where the ancillary input device
                                    handle is written.  MNULL if unsuccessful.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid ancillary input device index.  No hardware associated.
                LStatus_OUT_OF_RESOURCES    Specified ancillary input device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Ancillary input handle couldn't be returned because the ancillary input
                                            is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the ancillary input can't be
                  called.
                - If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that
                  modify the ancillary input can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at
                  any one time. If a handle with read/write access has already been granted to a caller,
                  any attempt to get another handle with read/write access returns a failure.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetHandle(
                LDevice_Handle          hDev,
                MUINT32                 uiIndex,
                LAccessMode             eAccessMode,
                LAncIn_Handle*        phAncIn);


/************************************************************************************************************\

Function:       LAncIn_ReleaseHandle

Description:    Releases a handle to an independent ancillary input device.

Parameters:     IN  hAncIn              Handle to the ancillary input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_ReleaseHandle(
                LAncIn_Handle     hAncIn);


/************************************************************************************************************\

Function:       LAncIn_EnumSignalType

Description:    Enumerates all supported ancillary input signal types.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  uiSignalIndex       Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No signal type value is available for the specified index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_EnumSignalType(
                LAncIn_Handle           hAncIn,
                MUINT32                 uiSignalIndex,
                LAnc_SignalType*        peSignalType);


/************************************************************************************************************\

Function:       LAncIn_GetDefaultSignalType

Description:    Gets the default signal type of the ancillary input device.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The default ancillary input signal type device may change after an input source detection.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetDefaultSignalType(
                LAncIn_Handle         hAncIn,
                LAnc_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAncIn_SetSignalType

Description:    Sets the ancillary input device signal type  of the ancillary input device.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  eSignalType         Ancillary signal type to set for the independent ancillary input.  Must be a
                                        value from the LAnc_SignalType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Setting the signal type may reserve and program some hardware resources that are shared
                  with other ancillary input device.

\************************************************************************************************************/
LAPI LStatus LAncIn_SetSignalType(
                LAncIn_Handle     hAncIn,
                LAnc_SignalType   eSignalType);


/************************************************************************************************************\

Function:       LAncIn_GetSignalType

Description:    Gets the current ancillary input device signal type.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetSignalType(
                LAncIn_Handle         hAncIn,
                LAnc_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAncIn_DetectSource

Description:    Performs an input source ancillary parameter detection based on the current signal type.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                OUT pbIsCapturable      Pointer to a MBOOL32 variable to be filled with MTRUE if the detected
                                        source can be captured and MFALSE if no source was detected or the
                                        detected source is outside the range of capturable sources.
                OUT pbIsDetected        Pointer to a MBOOL32 variable to be filled with MTRUE if a source,
                                        capturable or not, was detected and MFALSE otherwise.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because source detection is not supported by this
                                        ancillary input.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_DetectSource(
                LAncIn_Handle           hAncIn,
                MBOOL32*                pbIsDetected);


/************************************************************************************************************\

Function:       LAncIn_CreateBuffers

Description:    Creates ancillary capture buffers.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  uiBufferSize        Size, in byte, of the capture buffer.  Must be a multiple of ancillary
                                        frame size for PCM ancillary format capture buffers.
                IN  uiNbBuffers         Number of buffers allocated for the capture.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The ancillary input device allocates the specified number of buffers before the capture
                  is started.
                - The recommended minimum number of buffers is two (2) if the time taken to process a captured
                  buffer is less than the time taken to capture a buffer. Otherwise, a larger number of
                  buffers should be used.
                - The recommended number of buffers depends on the time the application needs to process the
                  captured buffers. There's no fixed maximum number of buffers; only the available memory
                  limits the maximum.
                - Extra buffers will be created for hardware processing. The number of extra buffers created
                  may be queried with LAncIn_GetCaptureState().


\************************************************************************************************************/
LAPI LStatus LAncIn_CreateBuffers(
                LAncIn_Handle   hAncIn,
                MUINT32         uiBufferSize,
                MUINT32         uiNbBuffers);


/************************************************************************************************************\

Function:       LAncIn_DestroyBuffers

Description:    Destroys the ancillary capture buffers.

Parameters:     IN  hAncIn              Handle to the ancillary input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - It's important to stop the capture and release the buffer before destroying the buffer.

\************************************************************************************************************/
LAPI LStatus LAncIn_DestroyBuffers(
                LAncIn_Handle     hAncIn);


/************************************************************************************************************\

Function:       LAncIn_GetBufferAttributes

Description:    Gets the current ancillary capture buffer attributes.

Parameters:     IN  hAncIn              Handle to ancillary input device.
                OUT puiBufferSize       Pointer to a MUINT32 variable to be filled with the size, in byte,
                                        of the capture buffer.
                OUT puiNbBuffers        Pointer to an MUINT32 variable to be filled with the number of
                                        buffers allowed to be used by the ancillary capture device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetBufferAttributes(
                LAncIn_Handle   hAncIn,
                MUINT32*        puiBufferSize,
                MUINT32*        puiNbBuffers);


/************************************************************************************************************\

Function:       LAncIn_StartCapture

Description:    Starts the ancillary capture.

Parameters:     IN  hAncIn              Handle to the ancillary input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The number of capture buffers the ancillary input device is allowed to use can be set via
                  LAncIn_CreateBuffers.

\************************************************************************************************************/
LAPI LStatus LAncIn_StartCapture(
                LAncIn_Handle hAncIn);


/************************************************************************************************************\

Function:       LAncIn_StartCaptureOnVideo

Description:    Starts the ancillary capture session synchronized with a video capture.

Parameters:     IN  hAncIn          Handle to the ancillary input device.
                IN  hVideoIn        Handle to the video input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Service used for the ancillary/video synchronization. It enables the synchronized capture of an
                  ancillary input and a video input to start.

\************************************************************************************************************/
LAPI LStatus LAncIn_StartCaptureOnVideo(
                LAncIn_Handle   hAncIn,
                LVideoIn_Handle hVideoIn);


/************************************************************************************************************\

Function:       LAncIn_StopCapture

Description:    Stops the ancillary capture.

Parameters:     IN  hAncIn              Handle to the ancillary input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - If capture is stopped in the middle of the capture of a buffer, this buffer is discarded.

\************************************************************************************************************/
LAPI LStatus LAncIn_StopCapture(
                LAncIn_Handle hAncIn);


/************************************************************************************************************\

Function:       LAncIn_GetNextBuffer

Description:    Gets the next ancillary buffer.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  bWaitBufferReady    MTRUE to wait for a Ready ancillary buffer, MFALSE to not wait.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and
                                        LINFINITE_TIMEOUT means wait forever.
                OUT phBuffer            Pointer to an LBuffer_Handle variable to be filled with the handle to
                                        ancillary buffer.
                OUT puiBufferCounter   Pointer to an MUINT64 variable to be filled with the buffer counter
                                        value when the returned buffer was captured or will be capture
                                        by the ancillary input.  Can be MNULL if this value is unused.
                OUT puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the
                                        returned buffer was captured by the ancillary input. Returns 0 when
                                        bWaitBufferReady is MFALSE and there's no Ready buffer.
                                        Can be MNULL if this value is unused.
                OUT pbIsTopField       Pointer to an MBOOL32 variable to be filled with MTRUE if the top
                                         field is being captured, MFALSE if it is the bottom field.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The ancillary capture buffers returned are allocated and owned by the ancillary input device.
                  They're loaned to the application for processing. The capture buffer state changes from
                  Released to Loaned when it's loaned to the application.
                - When bWaitBufferReady is set to MFALSE, the buffer is returned even if isn't Ready or
                  completely filled with the ancillary capture data. uiTickRefCounter is then set to 0.
                - The application can schedule ancillary processing in advance on Empty buffers or on parts of
                  Empty buffers in low latency mode. To get the handle of the Empty buffers, call the
                  LAncIn_GetNextBuffer service by setting bWaitBufferReady to MFALSE. The call will then
                  return the handle of an Empty capture buffer on which processing can be scheduled in a
                  device thread. To synchronize the processing and avoid using the Empty capture buffer before
                  it actually contains captured ancillary data, use LAncIn_ScheduleWaitCounter with the returned
                  value in puiCounter.
                - A loaned buffer must be released using the LAncIn_ReleaseBuffer or
                  LAncIn_ScheduleReleaseBuffer services after the application processed them to allow the
                  ancillary input device to reuse the buffers to capture new ancillary data.
                - If the capture buffers aren't released, the ancillary input device will run out of Empty buffers
                  and hold the last empty buffers available to capture until a buffer is released.
                - The counter is incremented each time a new valid signal from the input source is seen by the
                  ancillary input device, whether or not it's actually captured or dropped.
                - If two consecutive buffers, returned by the LAncIn_GetNextBuffer service, have a
                  difference greater than one between their associated counter values, some input source ancillary
                  data were dropped (i.e. have not been captured to capture buffers). This can occur if
                  capture buffers were unavailable at the time. To know how many input source ancillary data
                  haven't been captured, subtract 1 from the counter values difference.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetNextBuffer(
                LAncIn_Handle hAncIn,
                MBOOL32         bWaitBufferReady,
                MUINT32         uiTimeout_ms,
                LBuffer_Handle* phBuffer,
                MUINT64*        puiBufferCounter,
                MUINT64*        puiTickRefCounter,
                MBOOL32*        pbIsTopField);


/************************************************************************************************************\

Function:       LAncIn_ReleaseBuffer

Description:    Releases an ancillary capture buffer.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  hBuffer             Handle to the ancillary buffer to release.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture ancillary buffers loaned via the LAncIn_GetNextBuffer service must be released after
                  the application processes them to allow the ancillary input device to reuse them to capture
                  other input source ancillary data. The number of capture buffers for a given ancillary input device
                  is limited. Not releasing ancillary capture buffers starves the ancillary input device and will
                  eventually force it to temporarily stop capturing input source ancillary data.
                - The LAncIn_ReleaseBuffer service changes the states of a capture buffer to Empty and
                  Released.


\************************************************************************************************************/
LAPI LStatus LAncIn_ReleaseBuffer(
                LAncIn_Handle       hAncIn,
                LBuffer_Handle      hBuffer);


/************************************************************************************************************\

Function:       LAncIn_ScheduleReleaseBuffer

Description:    Schedules the release of an ancillary capture buffer of the ancillary input device by adding required
                commands to the passed in device thread object.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  hBuffer             Handle to the ancillary buffer to release.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - Capture ancillary buffers loaned via the LAncIn_GetNextBuffer service must be released after
                  the application processes them to allow the ancillary input device to reuse them to capture
                  other input source ancillary data. The number of capture buffers for a given ancillary input device
                  is limited. Not releasing ancillary capture buffers starves the ancillary input device and
                  eventually forces it to temporarily stop capturing input source ancillary data.
                - The LAncIn_ScheduleReleaseBuffer service changes the state of a capture buffer from Loaned
                  to Empty.
                - LStatus_TIMEOUT is returned when the ReleaseBuffer command can't be added to the device
                  thread because it's full and the timeout period expired before space became available.

\************************************************************************************************************/
LAPI LStatus LAncIn_ScheduleReleaseBuffer(
                LAncIn_Handle           hAncIn,
                LBuffer_Handle          hBuffer,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LAncIn_ScheduleWaitCounter

Description:    Schedules a wait for a counter of the ancillary input device by adding required commands to the
                passed in device thread object.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  uiCounter           Counter value when the buffer is captured by
                                        the ancillary input.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.


Comments:       - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread
                  because it's full and the timeout period expired before space became available.

\************************************************************************************************************/
LAPI LStatus LAncIn_ScheduleWaitCounter(
                LAncIn_Handle           hAncIn,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiCounter);


/************************************************************************************************************\

Function:       LAncIn_DiscardReadyBuffers

Description:    Discards all capture buffers in the Ready state.

Parameters:     IN  hAncIn              Handle to the ancillary input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture buffers in the Ready state contain captured data not yet retrieved by the
                  application.
                - The state of all discarded capture buffers is set to Empty. The capture buffers are also
                  marked as available to be used to capture new input source ancillary data.
                - The LAncIn_DiscardReadyBuffers service can be used to discard any unprocessed capture
                  buffers before re-starting the capture.
                - All capture buffers loaned to the application or currently in the Empty state are
                  unaffected by this service.

\************************************************************************************************************/
LAPI LStatus LAncIn_DiscardReadyBuffers(
                LAncIn_Handle hAncIn);


/************************************************************************************************************\

Function:       LAncIn_GetCaptureState

Description:    Return the ancillary capture state.

Parameters:     IN  hAncIn                  Handle to the ancillary input device.
                OUT peState                 Pointer to an LAncIn_State variable to be filled with
                                            the current capture state.
                OUT puiTotalBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                            allocated capture buffers count.
                OUT puiReadyBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                            ready capture buffers count.
                OUT puiLoanedBufferCount    Pointer to an MUINT32 variable to be filled with the currently
                                            loaned capture buffers count.
                OUT puiInternalBufferCount  Pointer to an MUINT32 variable to be filled with the number of
                                            internally created buffer to allow the hardware to run.
                                            These buffer are not part of the total buffer count.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - For more information on BufferCount, see LAncIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetCaptureState(
                LAncIn_Handle   hAncIn,
                LAnc_State*     peState,
                MUINT32*        puiTotalBufferCount,
                MUINT32*        puiReadyBufferCount,
                MUINT32*        puiLoanedBufferCount,
                MUINT32*        puiInternalBufferCount);


/************************************************************************************************************\

Function:       LAncIn_WaitForBufferChange

Description:    Blocks a CPU thread until the next buffer capture starts.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and
                                        LINFINITE_TIMEOUT means forever.
                OUTO puiCounter          Pointer to an MUINT64 variable to be filled with the number of
                                        the just started buffer. Can be MNULL if this value is unused.
                OUTO puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the
                                        counter was updated.  Can be MNULL if this value is unused.
                OUT  pbIsTopField       Pointer to an MBOOL32 variable to be filled with MTRUE if the top
                                         field is being captured, MFALSE if it is the bottom field.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_UNSUPPORTED     Function aborted because waiting for start is unsupported
                                        by the current hardware configuration.

Comments:       - For more information on counters, tick counters, and top ancillary data,
                  see LAncIn_GetLastBufferCounter.

\************************************************************************************************************/
LAPI LStatus LAncIn_WaitForBufferChange(
                LAncIn_Handle   hAncIn,
                MUINT32         uiTimeout_ms,
                MUINT64*        puiCounter,
                MUINT64*        puiTickRefCounter,
                MBOOL32*        pbIsTopField);


/************************************************************************************************************\

Function:       LAncIn_GetLastBufferCounter

Description:    Gets the last captured buffer counter value.

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                OUTO puiCounter         Pointer to an MUINT64 variable to be filled with the last counter
                                        number.  Can be MNULL if this value is unused.
                OUTO puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the
                                        counter was last updated.  Can be MNULL if this value is unused.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - For more information on counters, tick counters, and top ancillary data,
                  see LAncIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetLastBufferCounter(
                LAncIn_Handle   hAncIn,
                MUINT64*        puiCounter,
                MUINT64*        puiTickRefCounter);


/************************************************************************************************************\

Function:       LAncIn_GetCounterLUID

Description:    Gets captured counters unique identifiers (LUID).

Parameters:     IN  hAncIn              Handle to the ancillary input device.
                OUT puiCounterLUID      Pointer to an MUINT64 variable where the ancillary data counter unique
                                        identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Use to synchronizes the device thread with the completion of a counter.

\************************************************************************************************************/
LAPI LStatus LAncIn_GetCounterLUID(
                LAncIn_Handle        hAncIn,
                MUINT64*            puiCounterLUID);


/************************************************************************************************************\

Function:       LAncIn_WaitForEvent

Description:    Blocks a CPU thread until any of the specified capture events are signaled.

Parameters:     IN  hAncIn                  Handle to the ancillary input device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and
                                            LINFINITE_TIMEOUT means forever.
                OUT peSignaledEvent         Pointer to a LAncIn_Event variable to be filled with one
                                            signaled capture event.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Waiting for capture events isn't supported.
                LStatus_TIMEOUT             Function exited after a timeout.
                LStatus_CANCELED            Function exited. No capture in progress.

Comments:       - A capture event can be used to notify applications of important events during a capture
                   session.
                - Application can start waiting for event before the capture is started with
                   LAncIn_StartCapture.

\************************************************************************************************************/
LAPI LStatus LAncIn_WaitForEvent(
                LAncIn_Handle  hAncIn,
                MUINT32        uiTimeout_ms,
                LAncIn_Event*  peSignaledEvent);


#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LAncIn_H
